// mng4iectl.cpp
// Part of MNG4IE.
// Copyright 2002-2003 by Jason Summers

#include "stdafx.h"

#if defined(UNICODE) || defined(_UNICODE)
#error See comments in source code about UNICODE.
// This might compile with Unicode, but it hasn't really been tested,
// and there are probably some problems with it, some of which are
// marked with "possible unicode problem" comments.
// Choose a different configuration, or remove the line above if you want
// to try compiling with Unicode.
#endif

#include "mng4ie.h"
#include "bindstatuscallback.h"

// ---------------
// If you get an error about INT32 in jmorecfg.h, try modifying
// jmorecfg.h by changing:
//   typedef long INT32;
// to:
//   #ifdef JPEG_INTERNALS
//   typedef long INT32;
//   #endif
//   -----------
// If you get errors in lcms.h, try removing or renaming each
// occurrence of the FAR macro.
//   -----------
// Remove these #defines if necessary:
#define MNG_ERROR_TELLTALE
#define MNG_FULL_CMS
#define MNG_SUPPORT_DYNAMICMNG
//   -----------
#include <libmng.h>
// ---------------

#include "mng4iectl.h"
#include <hlink.h>
#include <commdlg.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <malloc.h>
#include <stdarg.h>

#include <jversion.h>      // part of libjpeg
#include <zlib.h>          // for zlibVersion


#define MNG4IE_CMS
//#define MNG4IE_TRACE

#define IDBASE           47000
#define ID_SAVEAS        (IDBASE+0)
#define ID_COPYIMAGE     (IDBASE+1)
#define ID_COPYURL       (IDBASE+2)
#define ID_VIEWIMAGE     (IDBASE+3)
#define ID_ABOUT         (IDBASE+4)
#define ID_FREEZE        (IDBASE+5)
#define ID_RESTARTANIM   (IDBASE+6)
#define ID_COPYLINKLOC   (IDBASE+7)
#define ID_STOPANIM      (IDBASE+8)
#define ID_SHOWERROR     (IDBASE+9)
#define ID_PROPERTIES    (IDBASE+10)


#ifdef MNG4IE_TRACE
static FILE *tracefile;
#endif

extern struct globals_struct globals;

/////////////////////////////////////////////////////////////////////////////
// Cmng4iectl

Cmng4iectl::Cmng4iectl()
{
	m_bWindowOnly = TRUE;
	m_force_bgcolor=0;
	m_bg_r = m_bg_g = m_bg_b = 0xff;
	m_fhWnd = NULL;

	// This will have to be deferred until ::Load
	m_scrolling=0;   //m_scrolling = (mode==NP_FULL);

	m_xscrollpos = m_yscrollpos = 0;
	m_windowwidth = m_windowheight = 0;
	m_loadstate = STATE_INIT;
	m_paintedyet = 0;
	m_mng=0;
	m_lpdib=NULL;
	lstrcpy(m_url,_T(""));
	m_frozen=0;
	m_needresume=0;
	m_errorflag=0;
	lstrcpy(m_errormsg,_T(""));
	m_dibsize = m_filesize = 0;
	lstrcpy(m_linkurl,_T(""));
	lstrcpy(m_linktarget,_T("_self"));
	m_islink=0;
	m_bkgdbrush=NULL;
	m_fullscreen=0;
	m_timer_set=0;
	m_timer2_set=0;
	m_dynamicmng= -1;
	m_mouse_over_mng=0;
	m_mouse_captured=0;
}


Cmng4iectl::~Cmng4iectl()
{

	// should any of this be moved to OnDestroy?

	if(m_mng) {
		m_dynamicmng=0;
		mng_cleanup(&m_mng); 
		m_mng=0;
	}
	if(m_lpdib) {
		free(m_lpdib);
		m_lpdib=NULL;
	}
	if(m_mngdata) {
		free(m_mngdata);
		m_mngdata=NULL;
		m_bytesalloc=0;
	}
	if(m_textdata) {
		free(m_textdata);
		m_textdata=NULL;
	}

	if(m_bkgdbrush) DeleteObject((HGDIOBJ)m_bkgdbrush);

}

STDMETHODIMP Cmng4iectl::GetInterfaceSafetyOptions( REFIID riid,
                                       DWORD *pdwSupportedOptions,
                                       DWORD *pdwEnabledOptions )
{
  *pdwSupportedOptions = INTERFACESAFE_FOR_UNTRUSTED_CALLER | INTERFACESAFE_FOR_UNTRUSTED_DATA;
  *pdwEnabledOptions = *pdwSupportedOptions;
  return S_OK;
}

STDMETHODIMP Cmng4iectl::SetInterfaceSafetyOptions( REFIID riid,
                                       DWORD dwOptionsSetMask,
                                       DWORD dwEnabledOptions )
{
  return S_OK;
}


HRESULT Cmng4iectl::OnDraw(ATL_DRAWINFO& di)
{
	RECT& rect = *(RECT*)di.prcBounds;
	RECT rect2;
	HDC hdc = di.hdcDraw;

	::SetWindowOrgEx(hdc,m_xscrollpos,m_yscrollpos,NULL);

	//::GetClientRect(hWnd,&rect);
	if(m_errorflag || !m_lpdib) {
		::SelectObject(hdc,globals.hfontMsg);
		::Rectangle(hdc,rect.left,rect.top,rect.right,rect.bottom);
		rect2.left=rect.left+2;
		rect2.top=rect.top+2;
		rect2.right=rect.right-2;
		rect2.bottom=rect.bottom-2;
		if(m_errorflag) {
			::DrawText(hdc,_T("MNG4IE ERROR!"),-1,&rect2,DT_LEFT|DT_WORDBREAK);
		}
		else {
			if(m_loadstate>=STATE_LOADING) {
				::DrawText(hdc,_T("MNG image loading..."),-1,&rect2,DT_LEFT|DT_WORDBREAK);
			}
			else {
				::DrawText(hdc,_T("MNG4IE"),-1,&rect2,DT_LEFT|DT_WORDBREAK);
			}
		}
	}
	else if(m_lpdib) {
		::StretchDIBits(hdc,
			0,0,m_lpdibinfo->biWidth,m_lpdibinfo->biHeight,
			0,0,m_lpdibinfo->biWidth,m_lpdibinfo->biHeight,
			&((BYTE*)(m_lpdib))[sizeof(BITMAPINFOHEADER)],
			(LPBITMAPINFO)m_lpdib,DIB_RGB_COLORS,SRCCOPY);
	}


	return S_OK;
}


LRESULT Cmng4iectl::OnCreate(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	m_fhWnd = m_hWnd;

	find_window_size();
	set_scrollbars();

	// It's easiest to wait till we have a window to start the
	// download, because the timer that the animation needs
	// uses a window.
	if(lstrlen(m_url)) {
		DownloadURL(CComBSTR(m_url));
	}
	//InvalidateRect( m_fhWnd, NULL, TRUE );
	//UpdateWindow( m_fhWnd );
	return 0;
}


LRESULT Cmng4iectl::OnDestroy(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	if(m_timer_set) {
		::KillTimer(m_fhWnd,1);
		m_timer_set=0;
	}
	if(m_timer2_set) {
		::KillTimer(m_fhWnd,2);
		m_timer2_set=0;
	}
	if(m_mouse_captured) {
		::ReleaseCapture();
		m_mouse_captured=0;
	}
	m_fhWnd = NULL;
	return 0;
}


STDMETHODIMP Cmng4iectl::DownloadURL(BSTR strURL)
{
	int propId;
	propId = 3456;

	COurBindStatusCallback<Cmng4iectl>::Download(this, OnData, strURL, 
		propId, m_spClientSite, TRUE);
	return S_OK;
}

int Cmng4iectl::ReadParamString(LPPROPERTYBAG pPropBag, LPERRORLOG pErrorLog,
								WCHAR *name, TCHAR *buf, int bufsize)
{
    VARIANT v;
	HRESULT hr;
	int retval=0;

	v.vt = VT_EMPTY; v.bstrVal = NULL;
	hr = pPropBag->Read(name, &v, pErrorLog);
	if(SUCCEEDED(hr)) {
		if(v.vt==VT_BSTR && v.bstrVal) {
			USES_CONVERSION;
			lstrcpyn(buf,OLE2T(v.bstrVal),bufsize);
			retval=1;
		}
		VariantClear(&v);
	}
	return retval;
}

static unsigned char gethex(const TCHAR *s)
{
	int v[2];
	int i;

	v[0]=v[1]=0; 
	for(i=0;i<2;i++) {
		if(s[i]>=_T('a') && s[i]<=_T('f')) v[i]=s[i]-87;
		if(s[i]>=_T('A') && s[i]<=_T('F')) v[i]=s[i]-55;
		if(s[i]>=_T('0') && s[i]<=_T('9')) v[i]=s[i]-48;
	}
	return (unsigned char)(v[0]*16+v[1]);
}

static void hexcolor2rgb(const TCHAR *s, unsigned int *r, unsigned int *g, unsigned int *b)
{
	if(lstrlen(s)!=7) return;
	if(s[0]!=_T('#')) return;
	(*r)= gethex(&s[1]);
	(*g)= gethex(&s[3]);
	(*b)= gethex(&s[5]);
}

// note that Load is called before OnCreate
// (at least, I *hope* it always is)
STDMETHODIMP Cmng4iectl::Load(LPPROPERTYBAG pPropBag, LPERRORLOG pErrorLog)
{
	TCHAR buf[500];

	// examine the <embed>/<param> tag arguments

	ReadParamString(pPropBag,pErrorLog,L"src",m_url,MAXLEN_URL);

	if(ReadParamString(pPropBag,pErrorLog,L"FULLSCREEN",buf,50)) {
		if(buf[0]=='Y' || buf[0]=='y') {
			m_fullscreen=1;
			m_scrolling=1;
		}
	}

	if(ReadParamString(pPropBag,pErrorLog,L"bgcolor",buf,50)) {
		hexcolor2rgb(buf,&m_bg_r,&m_bg_g,&m_bg_b);
	}
	else {
		OLE_COLOR clrBackColor;

		if (SUCCEEDED (GetAmbientBackColor (clrBackColor))) {
			m_bg_r = GetRValue(clrBackColor);
			m_bg_g = GetGValue(clrBackColor);
			m_bg_b = GetBValue(clrBackColor);
		}
	}

	m_bkgdbrush=CreateSolidBrush(RGB(m_bg_r,m_bg_g,m_bg_b));
	m_force_bgcolor=1; // this is always ON now. should probably remove it..

	if(ReadParamString(pPropBag,pErrorLog,L"href",m_linkurl,MAXLEN_URL)) {
		m_islink=1;
	}

	ReadParamString(pPropBag,pErrorLog,L"target",m_linktarget,MAXLEN_TARGET);
    return IPersistPropertyBagImpl<Cmng4iectl>::Load(pPropBag, pErrorLog);
}

#define MNG4IE_ALLOC_CHUNK_SIZE   131072
#define MNG4IE_MAX_FILE_SIZE      100000000

void Cmng4iectl::OnData(COurBindStatusCallback<Cmng4iectl>* pbsc, DWORD grfBSCF, BYTE* pBytes, DWORD dwSize)
{
	if(m_errorflag) return;
	if(m_loadstate>=STATE_LOADED) return;
	if(pbsc->m_propId != 3456) return; // wrong stream

	if(grfBSCF & BSCF_FIRSTDATANOTIFICATION) {
		m_libmngpos=0;
		m_bytesloaded=0;
		m_bytesalloc=0;
		m_byteswanted=0;
		m_mngdata=NULL;
		m_textdata=NULL;

		if(pbsc->m_dwTotalStreamSize > 0) {
			if(pbsc->m_dwTotalStreamSize > MNG4IE_MAX_FILE_SIZE) {
				set_error(_T("File too large"));
				return;
			}
			m_mngdata = (unsigned char*)malloc(pbsc->m_dwTotalStreamSize);
			m_bytesalloc= pbsc->m_dwTotalStreamSize;
		}
		my_init_mng();
		m_loadstate=STATE_LOADING;

	}

	if(m_bytesloaded+dwSize > (int)m_bytesalloc) {  // oops, overflowed our memory buffer
		if(m_bytesalloc > MNG4IE_MAX_FILE_SIZE) {
				set_error(_T("File too large"));
				return;
		}
		m_bytesalloc += MNG4IE_ALLOC_CHUNK_SIZE;
		if(m_mngdata) {
			m_mngdata= (unsigned char*)realloc((void*)m_mngdata, m_bytesalloc);
		}
		else {  // first time
			m_mngdata= (unsigned char*)malloc(m_bytesalloc);
		}
		if(!m_mngdata) {
			set_error(_T("Could not allocate memory for image (%d,%d,%p)"),m_bytesloaded,dwSize,pBytes);
			return;
		}
	}

	// now we should have enough room to copy the data to memory

	CopyMemory(&m_mngdata[m_bytesloaded],pBytes,dwSize);

	m_bytesloaded += dwSize;

	// now, check if it's time to call mng_read_resume
	if(m_needresume &&
		(m_bytesloaded >= (m_libmngpos + m_byteswanted)) )
	{
		m_needresume=0;

#ifdef MNG4IE_TRACE
	fprintf(tracefile,"NPP_Write display_resume bytesloaded=%d libmngpos=%d byteswanted=%d\n",
		m_bytesloaded,m_libmngpos,m_byteswanted);
#endif

		handle_read_error(mng_display_resume(m_mng) );
	}


	if(grfBSCF & BSCF_LASTDATANOTIFICATION) {
		m_filesize = m_bytesloaded;
		m_loadstate = STATE_LOADED;

#ifdef MNG4IE_TRACE
		fprintf(tracefile,"NPP_DestroyStream reason=%d needresume=%d\n",reason,m_needresume);
#endif

		if(m_needresume) {
			m_needresume=0;
			handle_read_error(mng_display_resume(m_mng) );
		}
	}
}

void Cmng4iectl::warn(TCHAR *fmt, ...)
{
	va_list ap;
	TCHAR buf[2048];

	va_start(ap, fmt);
	wvsprintf(buf,fmt, ap);
	va_end(ap);

	::MessageBox(m_fhWnd,buf,_T("MNG4IE"),MB_OK|MB_ICONWARNING);
}

void Cmng4iectl::set_error(TCHAR *fmt, ...)
{
	va_list ap;
	TCHAR buf[2048];

	va_start(ap, fmt);
	wvsprintf(buf,fmt, ap);
	va_end(ap);

	m_errorflag=1;
	lstrcpyn(m_errormsg,buf,256);

	if(m_lpdib) {
		free(m_lpdib);
		m_lpdib=NULL;
	}
	m_xscrollpos = m_yscrollpos = 0;

	if(m_fhWnd)
		::InvalidateRect(m_fhWnd,NULL,TRUE);
}


/* ----------------------------------------------------------------------- */
//    MNG callbacks

mng_ptr MNG4IE_CALLBACK Cmng4iectl::memallocfunc(mng_size_t n)
{
	return (mng_ptr) calloc(n,1);
}

void MNG4IE_CALLBACK Cmng4iectl::memfreefunc(mng_ptr p, mng_size_t n)
{
	free((void*)p);
}

mng_bool MNG4IE_CALLBACK Cmng4iectl::callback_openstream (mng_handle mng)
{
	return MNG_TRUE;
}

mng_bool MNG4IE_CALLBACK Cmng4iectl::callback_closestream (mng_handle mng)
{
	Cmng4iectl *This;
	This = (Cmng4iectl*) mng_get_userdata(mng);
	This->m_loadstate = STATE_LOADED;  // this is probably redundant

	return MNG_TRUE;
}

mng_bool MNG4IE_CALLBACK Cmng4iectl::callback_readdata (mng_handle mng,mng_ptr pBuf,
                      mng_uint32 Buflen,mng_uint32 *pRead)
{
	int n;
	Cmng4iectl *This;
	This = (Cmng4iectl*) mng_get_userdata(mng);

#ifdef MNG4IE_TRACE
	fprintf(tracefile,"readdata callback buflen=%d loadstate=%d bytesloaded=%d libmngpos=%d\n",
		Buflen,m_loadstate,m_bytesloaded, m_libmngpos);
#endif

	// do we have enough data available?
	if(This->m_bytesloaded - This->m_libmngpos >= Buflen) {
		CopyMemory(pBuf,&This->m_mngdata[This->m_libmngpos],Buflen);
		(*pRead)= Buflen;
		This->m_libmngpos += Buflen;

#ifdef MNG4IE_TRACE
	fprintf(tracefile,"returning full: %d\n",Buflen);
#endif
		This->m_byteswanted=0;
		return MNG_TRUE;
	}
	else if(This->m_loadstate>=STATE_LOADED) {
		// We don't have the data it asked for, but we're at the end
		// of file, so send it anyway...?

		n=This->m_bytesloaded-This->m_libmngpos;

		if(n>0) {
			CopyMemory(pBuf,&This->m_mngdata[This->m_libmngpos],n);
			This->m_libmngpos+=n;
		}
		(*pRead)=n;
		// so what do we return?
#ifdef MNG4IE_TRACE
	fprintf(tracefile,"returning partial: %d\n",n);
#endif
		This->m_byteswanted=0;
		return MNG_TRUE;
	}

	// else we don't yet have the data it's requesting
#ifdef MNG4IE_TRACE
	fprintf(tracefile,"returning 0\n");
#endif
	(*pRead)=0;
	This->m_byteswanted=Buflen;
	return MNG_TRUE;
}

mng_bool MNG4IE_CALLBACK Cmng4iectl::callback_processheader(mng_handle mng,mng_uint32 iWidth,mng_uint32 iHeight)
{
	Cmng4iectl *This;
	This = (Cmng4iectl*) mng_get_userdata(mng);

	This->m_diblinesize = (((iWidth * 24)+31)/32)*4;
	This->m_dibsize = sizeof(BITMAPINFOHEADER) + This->m_diblinesize*iHeight;
	This->m_lpdib = (unsigned char*)calloc(This->m_dibsize,1);
	This->m_lpdibinfo = (LPBITMAPINFOHEADER)This->m_lpdib;
	This->m_lpdibbits = &This->m_lpdib[sizeof(BITMAPINFOHEADER)];
	ZeroMemory((void*)This->m_lpdib,sizeof(BITMAPINFOHEADER));
	This->m_lpdibinfo->biSize = sizeof(BITMAPINFOHEADER);
	This->m_lpdibinfo->biWidth = iWidth;
	This->m_lpdibinfo->biHeight = iHeight;
	This->m_lpdibinfo->biPlanes = 1;
	This->m_lpdibinfo->biBitCount = 24;

	mng_set_canvasstyle (mng, MNG_CANVAS_BGR8);

	/*	if(m_fhWnd) {
		if((int)iWidth > m_windowwidth || (int)iHeight > m_windowheight) {
			m_scrolling=1;
		}
	} */

	This->set_scrollbars();
	return MNG_TRUE;
}

mng_ptr MNG4IE_CALLBACK Cmng4iectl::callback_getcanvasline (mng_handle mng, mng_uint32 iLinenr)
{
	unsigned char *pp;

	Cmng4iectl *This;
	This = (Cmng4iectl*) mng_get_userdata(mng);
	pp = (&This->m_lpdibbits[(This->m_lpdibinfo->biHeight-1-iLinenr)*This->m_diblinesize]);
	return (mng_ptr) pp;
}

mng_bool MNG4IE_CALLBACK Cmng4iectl::callback_refresh (mng_handle mng, mng_uint32 iLeft, mng_uint32 iTop,
                       mng_uint32 iRight, mng_uint32 iBottom)
{
	Cmng4iectl *This;
	RECT rect;

	This = (Cmng4iectl*) mng_get_userdata(mng);

	if(This->m_loadstate<STATE_VALIDFRAME) {
		This->m_loadstate=STATE_VALIDFRAME;
	}

	if(This->m_fhWnd) {
		if(This->m_paintedyet) {
			rect.left= iLeft      - This->m_xscrollpos;
			rect.top= iTop        - This->m_yscrollpos;
			rect.right= iLeft+iRight;
			rect.bottom= iTop+iBottom;

			::InvalidateRect(This->m_fhWnd,&rect,FALSE);
		}
		else {
			// Make sure the first paint clears the whole window
			::InvalidateRect(This->m_fhWnd,NULL,TRUE);
			This->m_paintedyet=1;
		}
		::UpdateWindow(This->m_fhWnd);
	}
	return MNG_TRUE;
}

mng_uint32 MNG4IE_CALLBACK Cmng4iectl::callback_gettickcount (mng_handle mng)
{
	return GetTickCount();
}


mng_bool MNG4IE_CALLBACK Cmng4iectl::callback_settimer (mng_handle mng,mng_uint32 iMsecs)
{
	Cmng4iectl *This;
	This = (Cmng4iectl*) mng_get_userdata(mng);

	if(This->m_timer_set) {
		::KillTimer(This->m_fhWnd,1);
		This->m_timer_set=0;
	}

	if(This->m_fhWnd) {
		if(!::SetTimer(This->m_fhWnd,1,(UINT)iMsecs,NULL)) {
			This->warn(_T("Unable to create a timer for animation"));
			This->m_frozen=1;
			//return MNG_FALSE;
			return MNG_TRUE;
		}
		This->m_timer_set=1;
	}
	return MNG_TRUE;
}

mng_bool MNG4IE_CALLBACK Cmng4iectl::callback_processtext(mng_handle mng,
    mng_uint8 iType, mng_pchar zKeyword, mng_pchar zText,
    mng_pchar zLanguage, mng_pchar zTranslation)
{
	Cmng4iectl *This;
	int pos,i;

	This = (Cmng4iectl*) mng_get_userdata(mng);

	if(!This->m_textdata) {
		This->m_textdata=(char*)malloc(MAXLEN_TEXT+10);
		if(!This->m_textdata) return MNG_TRUE;
		strcpy(This->m_textdata,"");
	}

	pos=strlen(This->m_textdata);
	if(pos>=(MAXLEN_TEXT-10)) return MNG_TRUE;

	if(pos>0) {    /* separate items with a blank line */
		This->m_textdata[pos++]='\r';
		This->m_textdata[pos++]='\n';
		This->m_textdata[pos++]='\r';
		This->m_textdata[pos++]='\n';
	}

	for(i=0;zKeyword[i];i++) {
		if(pos<MAXLEN_TEXT)
			This->m_textdata[pos++]=zKeyword[i];
	}
	This->m_textdata[pos++]=':';
	This->m_textdata[pos++]=' ';

	for(i=0;zText[i];i++) {
		if(pos<MAXLEN_TEXT) {
			if(zText[i]=='\n') {
				This->m_textdata[pos++]='\r';
			}
			This->m_textdata[pos++]=zText[i];
		}
	}
	This->m_textdata[pos++]='\0';

	return MNG_TRUE;
}

mng_bool MNG4IE_CALLBACK Cmng4iectl::callback_traceproc (mng_handle mng,
	mng_int32 iFuncnr,mng_int32 iFuncseq,mng_pchar zFuncname)
{
#ifdef MNG4IE_TRACE
	if(tracefile) {
		fprintf(tracefile,"%d\t%d\t%d\t%s\n",(int)mng,iFuncnr,iFuncseq,zFuncname);
	}
#endif
	return MNG_TRUE;
}

/* ----------------------------------------------------------------------- */

void Cmng4iectl::handle_read_error(mng_retcode rv)
{
	mng_int8     iSeverity;
	mng_chunkid  iChunkname;
	mng_uint32   iChunkseq;
	mng_int32    iExtra1;
	mng_int32    iExtra2;
	mng_pchar    zErrortext;

#ifdef MNG4IE_TRACE
	fprintf(tracefile,"returned: %d\n",rv);
#endif

	switch(rv) {
	case MNG_NOERROR: case MNG_NEEDTIMERWAIT:
		break;

	case MNG_NEEDMOREDATA:
		if(m_loadstate>=STATE_LOADED) {
			set_error(_T("Unexpected end of file"));
		}
		else {
			m_needresume=1;
		}
		break;

	case MNG_INVALIDSIG:
		set_error(_T("Invalid or missing MNG file"));
		break;

	default:

		mng_getlasterror(m_mng, &iSeverity,&iChunkname,&iChunkseq,&iExtra1,
			&iExtra2,&zErrortext);

		if(zErrortext) {
			set_error(_T("Error reported by libmng (%d)\r\n\r\n%s"),(int)rv,zErrortext);
		}
		else {
			set_error(_T("Error %d reported by libmng"),(int)rv);
		}
	}
}


int Cmng4iectl::init_color_management(void)
{
#ifdef MNG4IE_CMS
	mng_set_outputsrgb(m_mng);
#endif
	return 1;
}

// return 1 if okay
int Cmng4iectl::my_init_mng(void)
{
	mng_retcode rv;
	int err;

	m_mng = mng_initialize((mng_ptr)this,memallocfunc,memfreefunc,NULL);

#ifdef MNG4IE_CMS
	init_color_management();
#endif

	err=0;
	rv=mng_setcb_openstream    (m_mng, callback_openstream   ); if(rv) err++;
	rv=mng_setcb_closestream   (m_mng, callback_closestream  ); if(rv) err++;
	rv=mng_setcb_readdata      (m_mng, callback_readdata     ); if(rv) err++;
	rv=mng_setcb_processheader (m_mng, callback_processheader); if(rv) err++;
	rv=mng_setcb_getcanvasline (m_mng, callback_getcanvasline); if(rv) err++;
	rv=mng_setcb_refresh       (m_mng, callback_refresh      ); if(rv) err++;
	rv=mng_setcb_gettickcount  (m_mng, callback_gettickcount ); if(rv) err++;
	rv=mng_setcb_settimer      (m_mng, callback_settimer     ); if(rv) err++;
	rv=mng_setcb_processtext   (m_mng, callback_processtext  ); if(rv) err++;

#ifdef MNG4IE_TRACE
	rv=mng_setcb_traceproc     (m_mng, callback_traceproc    ); if(rv) err++;
#endif
	if(err) {
		warn(_T("Error setting libmng callback functions"));
		return 0;
	}

	rv= mng_set_suspensionmode (m_mng,MNG_TRUE);
	if(rv) {
		warn(_T("Error setting suspension mode"));
		return 0;
	}

	// if the web page author provided a bgcolor, use it
	if(m_force_bgcolor) {
		rv=mng_set_bgcolor (m_mng, (m_bg_r<<8)|m_bg_r,
			                       (m_bg_g<<8)|m_bg_g,
								   (m_bg_b<<8)|m_bg_b);
	}

#ifdef MNG4IE_TRACE
	fprintf(tracefile,"initial readdisplay\n");
#endif

	handle_read_error(mng_readdisplay(m_mng) );
	return 1;
}

void Cmng4iectl::find_window_size(void)
{
	RECT r;
	if(m_scrolling) {  // make sure scrollbars exist if needed
		::ShowScrollBar(m_fhWnd,SB_BOTH,TRUE);
	}
	::GetClientRect(m_fhWnd, &r);
	m_windowwidth=r.right;
	m_windowheight=r.bottom;
}

void Cmng4iectl::set_scrollbars(void)
{
	SCROLLINFO si;
	int maxpos;

	if(!m_scrolling) return;
	if(!m_fhWnd) return;

	ZeroMemory(&si,sizeof(SCROLLINFO));
	si.cbSize = sizeof(SCROLLINFO);

	// horizontal
	if(m_lpdib) {
		maxpos=m_lpdibinfo->biWidth-m_windowwidth;
		if(maxpos<0) maxpos=0;
		if(m_xscrollpos>maxpos) m_xscrollpos=maxpos;
		if(m_xscrollpos<0) m_xscrollpos=0;
		
		si.fMask = SIF_ALL|SIF_DISABLENOSCROLL;
		si.nMin  = 0;
		si.nMax  = m_lpdibinfo->biWidth -1;
		si.nPage = m_windowwidth;
		si.nPos  = m_xscrollpos;
	}
	else {  // no image to display
		si.fMask = SIF_ALL|SIF_DISABLENOSCROLL;
		si.nMin  = 0;
		si.nMax  = 0;
		si.nPage = 1;
		si.nPos  = 0;
	}
	::SetScrollInfo(m_fhWnd,SB_HORZ,&si,TRUE);

	// vertical
	if(m_lpdib) {
		maxpos=m_lpdibinfo->biHeight-m_windowheight;
		if(maxpos<0) maxpos=0;
		if(m_yscrollpos>maxpos) m_yscrollpos=maxpos;
		if(m_yscrollpos<0) m_yscrollpos=0;

		si.fMask = SIF_ALL|SIF_DISABLENOSCROLL;
		si.nMin  = 0;
		si.nMax  = m_lpdibinfo->biHeight -1;
		si.nPage = m_windowheight;
		si.nPos  = m_yscrollpos;
	}
	::SetScrollInfo(m_fhWnd,SB_VERT,&si,TRUE);
}

#define SCROLLLINE  40

void Cmng4iectl::scrollmsg(UINT msg,int code, short int pos)
{
	int page;
	int dx, dy;     // amount of scrolling
	int x_orig, y_orig;

	if(!m_scrolling) return;
	if(!m_lpdib) return;

	x_orig=m_xscrollpos;
	y_orig=m_yscrollpos;

	if(msg==WM_HSCROLL) {
		page=m_windowwidth-15;
		if(page<SCROLLLINE) page=SCROLLLINE;

		switch(code) {
		case SB_LINELEFT: m_xscrollpos-=SCROLLLINE; break;
		case SB_LINERIGHT: m_xscrollpos+=SCROLLLINE; break;
		case SB_PAGELEFT: m_xscrollpos-=page; break;
		case SB_PAGERIGHT: m_xscrollpos+=page; break;
		case SB_LEFT: m_xscrollpos=0; break;
		case SB_RIGHT: m_xscrollpos=m_lpdibinfo->biWidth; break;
		case SB_THUMBTRACK: m_xscrollpos=pos; break;
		default: return;
		}
		set_scrollbars();
	}
	else if(msg==WM_VSCROLL) {
		page=m_windowheight-15;
		if(page<SCROLLLINE) page=SCROLLLINE;

		switch(code) {
		case SB_LINEUP: m_yscrollpos-=SCROLLLINE; break;
		case SB_LINEDOWN: m_yscrollpos+=SCROLLLINE; break;
		case SB_PAGEUP: m_yscrollpos-=page; break;
		case SB_PAGEDOWN: m_yscrollpos+=page; break;
		case SB_TOP: m_yscrollpos=0; break;
		case SB_BOTTOM: m_yscrollpos=m_lpdibinfo->biHeight; break;
		case SB_THUMBTRACK: m_yscrollpos=pos; break;
		default: return;
		}
		set_scrollbars();
	}

	dx= x_orig - m_xscrollpos;
	dy= y_orig - m_yscrollpos;

	if(dx || dy) {  // if any change
		// GetClientRect(m_fhWnd,&cliprect);
		::ScrollWindowEx(m_fhWnd,dx,dy,NULL,NULL /*&cliprect*/,NULL,NULL,SW_INVALIDATE);
	}
}


/* Try to make a filename from the url. Caller must provide fn[MAX_PATH] buffer.
 * This function attempts to extract a bitmap filename from a URL,
 * but if it doesn't look like it contains an appropriate name,
 * it leaves it blank. */
static void url2filename(TCHAR *fn, TCHAR *url)
{
	int title,ext,i;

	lstrcpy(fn,_T(""));
	ext=0;   /* position of the file extention */
	title=0; /* position of the base filename */
	for(i=0;url[i];i++) {
		if(url[i]==_T('.') ) ext=i+1;
		if(url[i]==_T('/') ) title=i+1;
		if(url[i]==_T('\\')) title=i+1;
		if(url[i]==_T(':') ) title=i+1;
		if(url[i]==_T('=') ) title=i+1;
	}

	if (!_tcsicmp(&url[ext],_T("mng")) ||
		!_tcsicmp(&url[ext],_T("jng")) ||
		!_tcsicmp(&url[ext],_T("png")) )
	{
		lstrcpyn(fn,&url[title],MAX_PATH);
	}
}

// sanitize string and escape '&'s for use in a menu
static void escapeformenu(TCHAR *s1)
{
	int f, t, len;
	TCHAR s2[200];

	t=0;
	len=lstrlen(s1); if(len>50) len=50;
	for(f=0;f<len;f++) {
		if(s1[f]==_T('&')) {
			s2[t++]=_T('&');
			s2[t++]=_T('&');
		}
		else if(s1[f]<32) {     // fixme, possible unicode problem
			s2[t++]=_T('_');
		}
		else {
			s2[t++]=s1[f];
		}
	}
	s2[t]=_T('\0');
	lstrcpy(s1,s2);
}

static TCHAR *get_imagetype_name(mng_imgtype t)
{
	switch(t) {
	case mng_it_mng: return _T("MNG");
	case mng_it_png: return _T("PNG");
	case mng_it_jng: return _T("JNG");
	}
	return _T("Unknown");
}

/* Write the image to a local file  */
void Cmng4iectl::SaveImage(void)
{
	OPENFILENAME ofn;
	TCHAR fn[MAX_PATH];
	HANDLE hfile;
	BOOL b;
	mng_imgtype t;
	DWORD byteswritten;

	if(!m_mng || m_loadstate<STATE_LOADED ||
		m_bytesloaded != m_filesize)
	{
		warn(_T("Image not loaded -- can't save"));
		return;
	}

	if(lstrlen(m_url)) {
		url2filename(fn,m_url);
	}
	else {
		lstrcpy(fn,_T(""));
	}

	ZeroMemory(&ofn,sizeof(OPENFILENAME));
	ofn.lStructSize=sizeof(OPENFILENAME);
	ofn.hwndOwner=m_fhWnd;
	ofn.nFilterIndex=1;
	ofn.lpstrTitle=_T("Save Image As...");
	ofn.lpstrFile=fn;
	ofn.nMaxFile=MAX_PATH;
	ofn.Flags=OFN_PATHMUSTEXIST|OFN_HIDEREADONLY|OFN_OVERWRITEPROMPT;

	t=mng_get_sigtype(m_mng);
	if(t==mng_it_png) {
		ofn.lpstrDefExt=_T("png");  // FIXME also give an option of MNG
		ofn.lpstrFilter=_T("PNG (*.png)\0*.png\0\0");
	}
	else if(t==mng_it_jng) {
		ofn.lpstrDefExt=_T("jng");  // FIXME also give an option of MNG
		ofn.lpstrFilter=_T("JNG (*.jng)\0*.jng\0\0");
	}
	else {
		ofn.lpstrFilter=_T("MNG (*.mng)\0*.mng\0\0");
		ofn.lpstrDefExt=_T("mng");
	}

	if(GetSaveFileName(&ofn)) {
		// save to filename: ofn.lpstrFile
		hfile=CreateFile(ofn.lpstrFile,GENERIC_WRITE,FILE_SHARE_READ,
			NULL,CREATE_ALWAYS,FILE_ATTRIBUTE_NORMAL,NULL);
		if(hfile==INVALID_HANDLE_VALUE) {
			warn(_T("Unable to write file"));
		}
		else {
			b=WriteFile(hfile, m_mngdata, m_filesize,
				&byteswritten,NULL);
			if(!b || byteswritten != m_filesize) {
				warn(_T("Error writing file"));
			}
			CloseHandle(hfile);
		}
	}
}

void Cmng4iectl::CopyToClipboard(void *mem,int size,UINT format)
{
	HGLOBAL hClip;
	LPVOID lpClip;

	if(!mem) return;

	if(!::OpenClipboard(NULL)) {
		warn(_T("Can't open the clipboard"));
		return;
	}

	if(::EmptyClipboard()) {
		hClip=GlobalAlloc(GMEM_ZEROINIT|GMEM_MOVEABLE|GMEM_DDESHARE,size);
		lpClip=GlobalLock(hClip);
		if(lpClip) {
			CopyMemory(lpClip,mem,size);
			GlobalUnlock(hClip);
			if(! ::SetClipboardData(format,hClip)) {
				warn(_T("Can't set clipboard data"));
			}
		}
		else {
			warn(_T("Can't allocate memory for clipboard"));
		}
	}
	else {
		warn(_T("Can't clear the clipboard"));
	}
	::CloseClipboard();
}

void Cmng4iectl::AboutDialog(void)
{
	::DialogBoxParam(globals.hInstResource,_T("ABOUTDLG"),m_fhWnd,(DLGPROC)DlgProcAbout,(LPARAM)this);
}

void Cmng4iectl::PropDialog(void)
{
	::DialogBoxParam(globals.hInstResource,_T("PROPDLG"),m_fhWnd,(DLGPROC)DlgProcProp,(LPARAM)this);
}

void Cmng4iectl::display_last_error(void)
{
	if(m_errorflag) {
		warn(_T("%s"),m_errormsg);
	}
}

void Cmng4iectl::DynamicMNG_FireEvent(mng_uint8 eventtype, POINTS pos)
{
	mng_retcode r;
	if(!m_mng) return;
	if(m_dynamicmng ==  0) return;
	if(m_dynamicmng == -1) {
		r=mng_status_dynamic(m_mng);
		if(r==MNG_FALSE) {
			return;
		}
		else {
			m_dynamicmng=1;
		}
	}
	mng_trapevent(m_mng, eventtype, pos.x+m_xscrollpos, pos.y+m_yscrollpos);
}


LRESULT Cmng4iectl::OnSetCursor(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	if(LOWORD(lParam)==HTCLIENT) {
		if(m_islink) {
			::SetCursor(globals.hcurHandIE);
			return 1;
		}
	}
	return DefWindowProc(uMsg,wParam,lParam);
}

LRESULT Cmng4iectl::OnLButtonDown(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	::SetCapture(m_fhWnd);
	m_mouse_captured=1;

	if(m_dynamicmng && m_mng && !m_errorflag) {
		DynamicMNG_FireEvent(4,MAKEPOINTS(lParam));  // 4=mouse key down
	}
	return 0;
}

LRESULT Cmng4iectl::OnLButtonDblClk(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	return OnLButtonDown(uMsg,wParam,lParam,bHandled);
}

LRESULT Cmng4iectl::OnMouseActivate(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	return MA_ACTIVATE;
}

LRESULT Cmng4iectl::OnLButtonUP(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	RECT rc;
	POINT pt;

	if(m_mouse_captured) {
		::ReleaseCapture();
		m_mouse_captured=0;
	}

	if(m_dynamicmng && m_mng && !m_errorflag) {
		DynamicMNG_FireEvent(5,MAKEPOINTS(lParam));  // 5=mouse key up
	}

	// if mouse is not over image, don't follow links, etc.
	::GetWindowRect(m_hWnd,&rc);
	::GetCursorPos(&pt);
	if(!::PtInRect(&rc,pt)) return 0;

	if(m_islink) {
		USES_CONVERSION;
		HlinkSimpleNavigateToString(T2W(m_linkurl),NULL,
			T2W(m_linktarget),GetUnknown(),NULL,NULL,
			HLNF_INTERNALJUMP,0);
		return 0;
	}
	else if(m_errorflag) {
		display_last_error();
		return 0;
	}

	return 0;
}


LRESULT Cmng4iectl::OnMouseMove(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	POINTS pos;

	if(m_dynamicmng && m_mng && m_lpdib && !m_errorflag) {
		int overimage;

		pos=MAKEPOINTS(lParam);
		overimage=0;
		if(pos.x>=0 && pos.x<m_lpdibinfo->biWidth && pos.y>=0 && pos.y<m_lpdibinfo->biHeight) {
			overimage=1;
		}

		if(overimage) {
			if(m_mouse_over_mng) {
				// mouse is still over image: mouse move event
				DynamicMNG_FireEvent(2,pos);  // 2=mouse move
			}
			else {
				// mouse wasn't over the image but now it is: mouse-enter event
				DynamicMNG_FireEvent(1,pos); // mouse enter
			}
		}
		else { // mouse not now over image
			if(m_mouse_over_mng) { // ... but it used to be
				pos.x=0; pos.y=0;
				DynamicMNG_FireEvent(3,pos); // 3=mouse leave
			}
		}

		m_mouse_over_mng=overimage; // remember for next time

		if(m_mouse_over_mng && (m_dynamicmng==1) ) {
			// FIXME: Investigate using TrackMouseEvent() instead of
			// polling the cursor position.
#define MOUSE_POLL_INTERVAL  100    // milliseconds
			if(::SetTimer(m_fhWnd,2,MOUSE_POLL_INTERVAL,NULL)) {
				m_timer2_set=1;
			}
		}

		return 0;
	}
	return DefWindowProc(uMsg,wParam,lParam);
}

LRESULT Cmng4iectl::OnContextMenu(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	int cmd;
	HMENU menu;
	POINT pt;
	TCHAR buf[MAX_PATH], buf2[200];

	pt.x=0; pt.y=0;
	GetCursorPos(&pt);

	// create context menu dynamically
	menu=CreatePopupMenu();
	if(m_errorflag) {
		AppendMenu(menu,MF_ENABLED,ID_SHOWERROR,_T("SHOW ERROR MESSAGE"));
		AppendMenu(menu,MF_SEPARATOR,0,NULL);
	}

	AppendMenu(menu,(m_loadstate>=STATE_LOADED?MF_ENABLED:MF_GRAYED),ID_SAVEAS,_T("Save Image &As..."));
	AppendMenu(menu,(m_lpdib?MF_ENABLED:MF_GRAYED),ID_COPYIMAGE,_T("&Copy Image"));
	AppendMenu(menu,MF_ENABLED,ID_COPYURL,_T("Cop&y Image Location"));
	if(m_islink) {
		AppendMenu(menu,MF_ENABLED,ID_COPYLINKLOC,_T("Copy Link Location"));
	}

	url2filename(buf,m_url);
	escapeformenu(buf);
	if(lstrlen(buf)) {
		wsprintf(buf2,_T("View Image (%s)"),buf);
	}
	else {
		wsprintf(buf2,_T("View Image"));
	}
	AppendMenu(menu,MF_ENABLED,ID_VIEWIMAGE,buf2);
	AppendMenu(menu,MF_SEPARATOR,0,NULL);
	// AppendMenu(menu,(m_mng?MF_ENABLED:MF_GRAYED),ID_STOPANIM,"Stop Animation");
	AppendMenu(menu,(m_mng?MF_ENABLED:MF_GRAYED)|
		(m_frozen?MF_CHECKED:MF_UNCHECKED),ID_FREEZE,_T("&Freeze Animation"));
	// AppendMenu(menu,(m_mng?MF_ENABLED:MF_GRAYED),ID_RESTARTANIM,"Restart Animation");
	AppendMenu(menu,MF_SEPARATOR,0,NULL);

	AppendMenu(menu,MF_ENABLED,ID_PROPERTIES,_T("Properties..."));

	AppendMenu(menu,MF_ENABLED,ID_ABOUT,_T("About MNG4IE..."));

	cmd=TrackPopupMenuEx(menu, TPM_LEFTALIGN|TPM_TOPALIGN|TPM_NONOTIFY|TPM_RETURNCMD|
		TPM_RIGHTBUTTON,pt.x,pt.y,m_fhWnd,NULL);

	DestroyMenu(menu);

	switch(cmd) {

	//case ID_STOPANIM:
	//	if(m_mng) {
	//		::KillTimer(m_fhWnd,1);
	//		mng_display_freeze(m_mng);
	//	}
	//	break;

	case ID_FREEZE:  // freeze or unfreeze
		if(m_frozen) Animate(1);
		else Animate(0);

		break;

	//case ID_RESTARTANIM:
	//	if(!m_frozen) {
	//		::KillTimer(m_fhWnd,1);
	//		mng_display_freeze(m_mng);
	//	}
	//	m_frozen=1;
	//	mng_display_reset(m_mng);
	//	m_frozen=0;
	//	handle_read_error(mng_display_resume(m_mng) );
	//	break;

	case ID_SAVEAS:
		SaveImage();
		break;
	case ID_COPYIMAGE:
		if(m_lpdib) {
			CopyToClipboard((unsigned char*)m_lpdib,m_dibsize,CF_DIB);
		}
		else {
			warn(_T("No image to copy"));
		}
		break;
	case ID_COPYURL:
		CopyToClipboard(m_url,lstrlen(m_url)+1,CF_TEXT);
		break;
	case ID_COPYLINKLOC:
		if(m_islink) {
			CopyToClipboard(m_linkurl,lstrlen(m_linkurl)+1,CF_TEXT);
		}
		break;
	case ID_VIEWIMAGE:

		if(lstrlen(m_url)) {
			USES_CONVERSION;
			HlinkSimpleNavigateToString(T2W(m_url),NULL,
				L"_self",GetUnknown(),NULL,NULL,
				HLNF_INTERNALJUMP,0);
		}

		break;
	case ID_PROPERTIES:
		PropDialog();
		break;
	case ID_ABOUT:
		AboutDialog();
		break;
	case ID_SHOWERROR:
		display_last_error();
		break;
	}
	return 0;
}

BOOL CALLBACK Cmng4iectl::DlgProcProp(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
	TCHAR buf[4096],buf2[1024];

	switch(Msg) {
	case WM_INITDIALOG:
		{
			DWORD tabs[1];
			USES_CONVERSION;

			Cmng4iectl *This=(Cmng4iectl*)lParam;

			tabs[0]= 60;
			::SendDlgItemMessage(hWnd,IDC_IMGINFO,EM_SETTABSTOPS,(WPARAM)1,(LPARAM)tabs);

			wsprintf(buf,_T("URL:\t%s\r\n"),This->m_url);

			if(This->m_lpdib) {
				wsprintf(buf2,_T("Dimensions:\t%d x %d\r\n"),This->m_lpdibinfo->biWidth,
					This->m_lpdibinfo->biHeight);
				lstrcat(buf,buf2);
			}

			if(This->m_lpdib && !This->m_fullscreen) {
				wsprintf(buf2,_T("Window:\t%d x %d\r\n"),This->m_windowwidth,
					This->m_windowheight);
				lstrcat(buf,buf2);
			}

			if(This->m_filesize) {
				wsprintf(buf2,_T("File size:\t%u bytes\r\n"),This->m_filesize);
				lstrcat(buf,buf2);
			}

#ifdef _DEBUG
			if(This->m_mngdata && This->m_lpdib && This->m_bytesalloc && This->m_dibsize) {
				// note this doesn't include memory used by libmng
				wsprintf(buf2,_T("Memory used:\t%u bytes\r\n"),
					This->m_bytesalloc + This->m_dibsize);
				lstrcat(buf,buf2);
			}
#endif

			if(This->m_islink) {
				wsprintf(buf2,_T("Link to:\t%s\r\n"),This->m_linkurl);
				lstrcat(buf,buf2);
				if(lstrcmp(This->m_linktarget,_T("_self"))) {
					wsprintf(buf2,_T("Link target:\t%s\r\n"),This->m_linktarget);
					lstrcat(buf,buf2);
				}
			}

			if(This->m_loadstate >= STATE_VALIDFRAME) {
				wsprintf(buf2,_T("Signature:\t%s\r\n"),get_imagetype_name(mng_get_sigtype(This->m_mng)));
				lstrcat(buf,buf2);
				wsprintf(buf2,_T("Image type:\t%s\r\n"),get_imagetype_name(mng_get_imagetype(This->m_mng)));
				lstrcat(buf,buf2);
				wsprintf(buf2,_T("Simplicity:\t0x%08x\r\n"),mng_get_simplicity(This->m_mng));
				lstrcat(buf,buf2);
				wsprintf(buf2,_T("Frame count:\t%u\r\n"),mng_get_framecount(This->m_mng));
				lstrcat(buf,buf2);
				wsprintf(buf2,_T("Layer count:\t%u\r\n"),mng_get_layercount(This->m_mng));
				lstrcat(buf,buf2);
				wsprintf(buf2,_T("Play time:\t%u\r\n"),mng_get_playtime(This->m_mng));
				lstrcat(buf,buf2);
			}

			::SetDlgItemText(hWnd,IDC_IMGINFO,buf);

			if(This->m_textdata)
				::SetDlgItemText(hWnd,IDC_MNGTEXT,A2T(This->m_textdata));
		}
		return(TRUE);
	case WM_CLOSE:
		::EndDialog(hWnd,0);
		return(TRUE);
	case WM_COMMAND:
		switch(wParam) {
		case IDOK:
		case IDCANCEL:
			::EndDialog(hWnd,0);
			return(TRUE);
		}
	}
	return(FALSE);
}

BOOL CALLBACK Cmng4iectl::DlgProcAbout(HWND hWnd, UINT Msg, WPARAM wParam, LPARAM lParam)
{
	TCHAR buf[4096],buf2[1024],buf3[300];

	switch(Msg) {
	case WM_INITDIALOG:
		{
			//DWORD tabs[1];

			Cmng4iectl *This=(Cmng4iectl*)lParam;

			//tabs[0]= 60;
			//::SendDlgItemMessage(hWnd,IDC_IMGINFO,EM_SETTABSTOPS,(WPARAM)1,(LPARAM)tabs);

			wsprintf(buf,_T("MNG4IE ActiveX control, Version %s\r\n%s")
#ifdef _DEBUG
				_T(" DEBUG BUILD")
#endif
				_T("\r\nCopyright (C) 2002-2003 by Jason Summers\r\n\r\n"),MNG4IEVERS,__DATE__);

			wsprintf(buf2,_T("Based on libmng by Gerard Juyn.\r\n"));
			lstrcat(buf,buf2);

			wsprintf(buf2,_T("libmng version: %s\r\n\r\n"),mng_version_text());  // fixme, possible unicode problem
			lstrcat(buf,buf2);

			wsprintf(buf2,_T("Uses the zlib compression library.\r\n"));
			lstrcat(buf,buf2);
			wsprintf(buf2,_T("zlib version: %s\r\n\r\n"),zlibVersion());  // fixme, possible unicode problem
			lstrcat(buf,buf2);

			wsprintf(buf2,_T("This software is based in part on the work of the ")
				_T("Independent JPEG Group.\r\n"));
			lstrcat(buf,buf2);
			wsprintf(buf2,_T("IJG JPEG library version: %s\r\n%s\r\n\r\n"),JVERSION,JCOPYRIGHT);  // fixme possible unicode problem
			lstrcat(buf,buf2);

#ifdef MNG4IE_CMS
			wsprintf(buf2,_T("Uses the lcms color management library by Mart Maria. ")
				_T("lcms is distributed under the terms of the GNU LESSER GENERAL PUBLIC LICENSE.\r\n\r\n"));
			lstrcat(buf,buf2);
#endif
			if(::GetModuleFileName(globals.hInst,buf3,260)) {
				wsprintf(buf2,_T("MNG4IE location: %s\r\n"),buf3);
				lstrcat(buf,buf2);
			}

			::SetDlgItemText(hWnd,IDC_PRGINFO,buf);

		}
		return(TRUE);
	case WM_CLOSE:
		::EndDialog(hWnd,0);
		return(TRUE);
	case WM_COMMAND:
		switch(wParam) {
		case IDOK:
		case IDCANCEL:
			::EndDialog(hWnd,0);
			return(TRUE);
		}
	}
	return(FALSE);
}


LRESULT Cmng4iectl::OnSize(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	find_window_size();
	set_scrollbars();
	return 0;
}

LRESULT Cmng4iectl::OnHScroll(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	scrollmsg(uMsg,(int)(LOWORD(wParam)),(short int)(HIWORD(wParam)));
	return 0;
}

LRESULT Cmng4iectl::OnVScroll(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	scrollmsg(uMsg,(int)(LOWORD(wParam)),(short int)(HIWORD(wParam)));
	return 0;
}


LRESULT Cmng4iectl::OnTimer(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	switch(wParam) {
	case 1: // the main animation timer
		::KillTimer(m_hWnd,1);
		m_timer_set=0;

#ifdef MNGPLG_TRACE
		fprintf(tracefile,"WM_TIMER display_resume bytesloaded=%d\n",m_bytesloaded);
#endif

		if(m_mng) {
			if(!m_needresume) {
				handle_read_error(mng_display_resume(m_mng) );
			}
		}
		return 0;

	case 2: // timer for polling mouse position
		RECT rc;
		POINT pt;
		::GetWindowRect(m_hWnd,&rc);
		::GetCursorPos(&pt);
		if(!::PtInRect(&rc,pt)) {
			POINTS pos;
			::KillTimer(m_hWnd,2);
			m_timer2_set=0;
			pos.x=0; pos.y=0;
			DynamicMNG_FireEvent(3,pos); // 3=mouse leave
			m_mouse_over_mng=0;
		}
		return 0;
	}

	return DefWindowProc(uMsg,wParam,lParam);
}

LRESULT Cmng4iectl::OnEraseBkgnd(UINT uMsg, WPARAM wParam, LPARAM lParam, BOOL& bHandled)
{
	HBRUSH br;
	HDC hdc;
	RECT rect;

	hdc= (HDC)wParam;

	if(m_bkgdbrush)
		br=m_bkgdbrush;
	else
		br= (HBRUSH)::GetStockObject(GRAY_BRUSH);

	::GetClientRect(m_fhWnd,&rect);
	::FillRect(hdc,&rect,br);
	return 1;

}

STDMETHODIMP Cmng4iectl::get_Version(long *pVal)
{
	*pVal = MNG4IEVERS_INT;
	return S_OK;
}

STDMETHODIMP Cmng4iectl::Animate(long x)
{
	if(!m_mng) return S_OK;

	if(x) {  // unfreeze
		if(!m_frozen) return S_OK;
		m_frozen=0;
		handle_read_error(mng_display_resume(m_mng) );
	}
	else { // freeze
		if(m_frozen) return S_OK;
		m_frozen=1;
		if(m_timer_set) {
			::KillTimer(m_fhWnd,1);
			m_timer_set=0;
		}
		mng_display_freeze(m_mng);
	}

	return S_OK;
}
